// ==UserScript==
// @name         5ch サムネイル表示（画像専用）
// @namespace    https://example.com/
// @version      1.3.1
// @description  5ch投稿内の画像（jpg, png, gifなど）をサムネイル＆ポップアップ表示。ポップアップ画像はドラッグ移動＆Ctrl+マウスホイールでズーム可能。ドラッグ後に閉じるのを防止。
// @match        *://*.5ch.net/test/read.cgi/*
// @grant        none
// ==/UserScript==

(function () {
    'use strict';

    const imageExtensions = ['.jpg', '.jpeg', '.png', '.gif', '.bmp', '.webp'];
    const jumpPrefix = 'http://jump.5ch.net/?';
    const PROCESSED_ATTR = 'data-image-thumbnail-processed';

    let imageUrls = [];
    let currentIndex = -1;

    // 拡張：クエリ付き画像URLにも対応
    function isImageUrl(url) {
        try {
            const u = new URL(url);
            const pathname = u.pathname.toLowerCase();
            const formatParam = u.searchParams.get('format')?.toLowerCase();

            if (imageExtensions.some(ext => pathname.endsWith(ext))) return true;
            if (formatParam && imageExtensions.some(ext => formatParam === ext.slice(1))) return true;

            return false;
        } catch (e) {
            return false;
        }
    }

    function extractUrls(post) {
        return [...new Set([...post.querySelectorAll('a[href]')].map(a => {
            const href = a.getAttribute('href');
            return href?.startsWith(jumpPrefix) ? href.slice(jumpPrefix.length) : href;
        }))];
    }

    function createOverlay() {
        const overlay = document.createElement('div');
        overlay.style.position = 'fixed';
        overlay.style.top = 0;
        overlay.style.left = 0;
        overlay.style.width = '100vw';
        overlay.style.height = '100vh';
        overlay.style.backgroundColor = 'rgba(0,0,0,0.85)';
        overlay.style.zIndex = 999999;
        overlay.style.display = 'flex';
        overlay.style.justifyContent = 'center';
        overlay.style.alignItems = 'center';
        overlay.style.cursor = 'pointer';
        return overlay;
    }

    function createArrow(direction, clickHandler) {
        const arrow = document.createElement('div');
        arrow.style.position = 'absolute';
        arrow.style.top = '50%';
        arrow.style[direction === 'next' ? 'right' : 'left'] = '20px';
        arrow.style.transform = 'translateY(-50%)';
        arrow.style.cursor = 'pointer';
        arrow.style.zIndex = '1000000';

        const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
        svg.setAttribute('width', '40');
        svg.setAttribute('height', '40');
        svg.setAttribute('viewBox', '0 0 24 24');
        svg.setAttribute('fill', 'none');
        svg.setAttribute('stroke', '#fff');
        svg.setAttribute('stroke-width', '2');
        svg.setAttribute('stroke-linecap', 'round');
        svg.setAttribute('stroke-linejoin', 'round');

        const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
        path.setAttribute('d', direction === 'next' ? 'M9 18l6-6-6-6' : 'M15 6l-6 6 6 6');
        svg.appendChild(path);
        arrow.appendChild(svg);

        arrow.addEventListener('click', (e) => {
            e.stopPropagation();
            clickHandler();
        });

        return arrow;
    }

    function openImagePopup(url) {
        const overlay = createOverlay();

        const img = document.createElement('img');
        img.src = url;
        img.style.maxWidth = '90vw';
        img.style.maxHeight = '90vh';
        img.style.borderRadius = '6px';
        img.style.position = 'fixed';
        img.style.top = '50%';
        img.style.left = '50%';
        img.style.cursor = 'grab';
        img.style.userSelect = 'none';

        let translateX = 0;
        let translateY = 0;
        let scale = 1;

        updateTransform();

        let isDragging = false;
        let startX, startY;
        let origX, origY;

        img.addEventListener('mousedown', e => {
            e.preventDefault();
            isDragging = true;
            startX = e.clientX;
            startY = e.clientY;
            origX = translateX;
            origY = translateY;
            img.style.cursor = 'grabbing';
        });

        window.addEventListener('mousemove', e => {
            if (!isDragging) return;
            e.preventDefault();
            const dx = e.clientX - startX;
            const dy = e.clientY - startY;
            translateX = origX + dx;
            translateY = origY + dy;
            updateTransform();
        });

        window.addEventListener('mouseup', () => {
            if (isDragging) {
                isDragging = false;
                img.style.cursor = 'grab';
            }
        });

        img.addEventListener('wheel', e => {
            e.preventDefault();
            const zoomStep = 0.1;

            if (e.ctrlKey) {
                if (e.deltaY < 0) {
                    scale = Math.min(scale + zoomStep, 5);
                } else {
                    scale = Math.max(scale - zoomStep, 0.2);
                }
                updateTransform();
            } else {
                if (e.deltaY > 0) showNextImage();
                else if (e.deltaY < 0) showPrevImage();
            }
        });

        img.addEventListener('click', e => e.stopPropagation());
        overlay.addEventListener('click', () => document.body.removeChild(overlay));

        const showNextImage = () => {
            if (currentIndex < imageUrls.length - 1) {
                currentIndex++;
                img.src = imageUrls[currentIndex];
            }
        };

        const showPrevImage = () => {
            if (currentIndex > 0) {
                currentIndex--;
                img.src = imageUrls[currentIndex];
            }
        };

        overlay.appendChild(createArrow('prev', showPrevImage));
        overlay.appendChild(createArrow('next', showNextImage));

        function updateTransform() {
            img.style.transform = `translate(calc(-50% + ${translateX}px), calc(-50% + ${translateY}px)) scale(${scale})`;
        }

        overlay.appendChild(img);
        document.body.appendChild(overlay);
    }

    function addImageThumbnails(post) {
        if (post.hasAttribute(PROCESSED_ATTR)) return;

        const wrapper = document.createElement('div');
        post.appendChild(wrapper);

        extractUrls(post).forEach(url => {
            if (!url || !isImageUrl(url)) return;

            const thumb = document.createElement('img');
            thumb.src = url;
            thumb.style.maxWidth = '120px';
            thumb.style.maxHeight = '90px';
            thumb.style.margin = '12px';
            thumb.style.cursor = 'pointer';
            thumb.style.border = '1px solid #ccc';
            thumb.style.borderRadius = '4px';

            thumb.addEventListener('click', () => {
                imageUrls = [];
                document.querySelectorAll('.post-content').forEach(content => {
                    imageUrls.push(...extractUrls(content).filter(isImageUrl));
                });
                currentIndex = imageUrls.indexOf(url);
                openImagePopup(url);
            });

            wrapper.appendChild(thumb);
        });

        post.setAttribute(PROCESSED_ATTR, 'true');
    }

    const observer = new IntersectionObserver(entries => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                addImageThumbnails(entry.target);
                observer.unobserve(entry.target);
            }
        });
    }, { threshold: 0.1 });

    function observePosts() {
        document.querySelectorAll('.post-content').forEach(post => {
            if (!post.hasAttribute(PROCESSED_ATTR)) {
                observer.observe(post);
            }
        });
    }

    window.addEventListener('load', observePosts);
    setInterval(observePosts, 3000);
})();
